/**
 * Проверка доступа по админской таблице (лист «Оплаты»).
 * Используется только на сервере, не передаётся в UI и настройки пользователя.
 */
const fs = require('fs');
const path = require('path');
const { GoogleSpreadsheet } = require('google-spreadsheet');
const { JWT } = require('google-auth-library');

const ADMIN_SHEET_TAB = process.env.ADMIN_SHEET_TAB_TITLE || 'Оплаты';
const STATUS_OK = 'Доступно';

function loadCreds() {
  const email = (process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL || '').trim();
  const key = (process.env.GOOGLE_PRIVATE_KEY || '').trim();
  if (email && key) {
    return { client_email: email, private_key: key };
  }
  const credsPath = process.env.GOOGLE_CREDENTIALS_PATH || './credentials.json';
  const p = path.isAbsolute(credsPath) ? credsPath : path.join(process.cwd(), credsPath);
  if (!fs.existsSync(p)) {
    throw new Error('Файл учётных данных Google не найден. Проверьте GOOGLE_CREDENTIALS_PATH.');
  }
  const raw = fs.readFileSync(p, 'utf8');
  const json = JSON.parse(raw);
  if (!json.client_email || !json.private_key) {
    throw new Error('Неверный формат credentials.json');
  }
  return json;
}

/**
 * Проверяет, есть ли у таблицы с id sheetId доступ (статус «Доступно» в админской таблице).
 * @param {string} sheetId — ID Google Таблицы (из ссылки)
 * @returns {Promise<boolean>}
 */
async function checkSheetAccess(sheetId) {
  const adminId = (process.env.ADMIN_SHEET_ID || '').trim();
  if (!adminId) {
    console.warn('[admin-access] ADMIN_SHEET_ID не задан, доступ не проверяется.');
    return true;
  }
  const target = (sheetId || '').trim();
  if (!target) return false;

  const creds = loadCreds();
  const auth = new JWT({
    email: creds.client_email,
    key: creds.private_key,
    scopes: ['https://www.googleapis.com/auth/spreadsheets']
  });

  const doc = new GoogleSpreadsheet(adminId, auth);
  await doc.loadInfo();

  const sheet = doc.sheetsByTitle[ADMIN_SHEET_TAB] || doc.sheetsByIndex[0];
  if (!sheet) return false;

  await sheet.loadHeaderRow();
  const rows = await sheet.getRows();

  for (const row of rows) {
    const id = (row.Id != null ? String(row.Id) : (row.id != null ? String(row.id) : '')).trim();
    const status = (row.Статус != null ? String(row.Статус) : (row['Статус'] != null ? String(row['Статус']) : '')).trim();
    if (id === target && status === STATUS_OK) {
      return true;
    }
  }
  return false;
}

module.exports = { checkSheetAccess };
