const { chromium } = require('playwright');

const STEALTH_ARGS = [
  '--disable-blink-features=AutomationControlled',
  '--disable-dev-shm-usage',
  '--no-sandbox',
  '--disable-infobars',
  '--ignore-certificate-errors'
];

const DEFAULT_VIEWPORT = { width: 1400, height: 900 };
const DEFAULT_LOCALE = 'ru-RU';

/**
 * Возвращает { browser, context }.
 * - userDataDir: запуск с папкой профиля (Chrome/Яндекс) — сессия и авторизация из профиля, без CDP.
 * - browserCdpUrl: подключение к уже запущенному браузеру по отладочному порту.
 * - иначе: обычный запуск Chromium.
 */
async function createBrowserAndContext(cfg) {
  const { userDataDir, browserCdpUrl, headless, slowMoMs } = cfg;

  if (userDataDir) {
    try {
      const context = await chromium.launchPersistentContext(userDataDir, {
        headless: Boolean(headless),
        slowMo: Number(slowMoMs) || 0,
        locale: DEFAULT_LOCALE,
        viewport: DEFAULT_VIEWPORT,
        args: STEALTH_ARGS,
        ignoreDefaultArgs: ['--enable-automation']
      });
      return { browser: null, context };
    } catch (err) {
      const msg = err && err.message ? err.message : String(err);
      if (
        msg.includes('ProcessSingleton') ||
        msg.includes('profile is already in use') ||
        msg.includes('Lock file')
      ) {
        throw new Error(
          'Профиль браузера занят. Полностью закройте Яндекс.Браузер (все окна и иконку в трее). ' +
            'Если не помогло — откройте Диспетчер задач (Ctrl+Shift+Esc) и завершите все процессы «Браузер» или «Yandex». Затем снова: npm start'
        );
      }
      throw err;
    }
  }

  if (browserCdpUrl) {
    try {
      const browser = await chromium.connectOverCDP(browserCdpUrl, { timeout: 15000 });
      const contexts = browser.contexts();
      const context = contexts[0];
      if (!context) {
        await browser.close().catch(() => {});
        throw new Error(
          'К браузеру подключились, но контекст не найден. Запустите браузер с флагом --remote-debugging-port=9222'
        );
      }
      return { browser, context };
    } catch (err) {
      const msg = err && err.message ? err.message : String(err);
      if (msg.includes('ECONNREFUSED') || msg.includes('connect')) {
        throw new Error(
          `Не удалось подключиться к браузеру по адресу ${browserCdpUrl}. Закройте все окна браузера и запустите его с отладочным портом, либо используйте USER_DATA_DIR (профиль).`
        );
      }
      throw err;
    }
  }

  const browser = await chromium.launch({
    headless: Boolean(headless),
    slowMo: Number(slowMoMs) || 0,
    args: STEALTH_ARGS,
    ignoreDefaultArgs: ['--enable-automation']
  });
  const context = await browser.newContext({
    locale: DEFAULT_LOCALE,
    viewport: DEFAULT_VIEWPORT,
    userAgent:
      'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    extraHTTPHeaders: { 'Accept-Language': 'ru-RU,ru;q=0.9,en;q=0.8' }
  });
  return { browser, context };
}

module.exports = { createBrowserAndContext };
